#ifndef _SESSION_H
#define _SESSION_H

#include "UnityNP.h"
#include "Mutex.h"

namespace UnityPlugin
{
	class SessionAttribute
	{
	public:
		const char* name;
		const char* binValue;
		int intValue;
		int searchOperator;
	};

	class ModifySessionAttribute
	{
	public:
		const char* name;
		const char* binValue;
		int intValue;
	};

	struct SessionInfo
	{
		const char* sessionName;	// Session name.
		int sessionID;				// The ID of this session, use this ID when joining a session.
		int maxMembers;				// The maximum numbers of members in the session.
		int numMembers;				// The current number of members in the session.
		int numSessionAttributes;	// The number of attributes in the session. 
		int reservedSlots;			// The number of slots that are reserved for a friend in a session.
		int openSlots;				// The number of slots that are open to the public.
		int worldId;				// The world ID for a session. 
		int serverId;				// The server ID for a session.
		int matchingContext;		//  The Matching2 context ID. This should be used when using the %NP Matching2 library chat function to send messages to members in the current session.
		UInt64 roomId;				// The ID of the current session.
	};

	struct SessionAttributeInfo
	{
		const char* attribute;			// The name of the session attribute.
		int attributeType;				// The type of the attribute.
		int searchOperator;				// The search operator if the session attribute type is SCE_SESSION_SEARCH_ATTRIBUTE_*. This is used when filtering the sessions.
		int maxSize;					// The size of the data in attributeValue.
		int attributeValueType;			// The type of value contained in attributeValue, SCE_SESSION_ATTRIBUTE_VALUE_* flags.
		int attributeIntValue;			// The integer value of a session attribute. 
		const char* attributeBinValue;	// The binary data of a session attribute.
	};

	struct SessionMemberInfo
	{
		const unsigned char* npID;	// NpID, contains data corresponding to the SceNpID structure, can be used as id to send to when messaging.
		int npIDSize;				// Size of the npID byte array.
		const char* npOnlineID;		// NpID, just the string part of the ID, 16 bytes plus terminator.
		int memberId;				// The room member ID of the member.
		int natType;				// The member's NAT Type.
		int memberFlag;				// Specifies whether this member is the owner, another member or the current user.
		UInt64 joinDate;			// The date and time the user joined the session.
		int addr;					// The IP address. 
		int port;					// The port number.
	};

	// Session status...
	PRX_EXPORT bool PrxSessionIsBusy();
	PRX_EXPORT bool PrxSessionGetLastError(ResultCode* result);
	PRX_EXPORT bool PrxSessionInSession();

	// Game session attributes.
	PRX_EXPORT ErrorCode PrxSessionClearAttributeDefinitions();
	PRX_EXPORT ErrorCode PrxSessionAddAttributeDefinition(const char* name, int type, int valueType, int maxSize);
	PRX_EXPORT ErrorCode PrxSessionRegisterAttributeDefinitions();

	// Session creation...
	PRX_EXPORT ErrorCode PrxSessionClearAttributes();
	PRX_EXPORT ErrorCode PrxSessionAddAttribute(SessionAttribute* attribute);
	PRX_EXPORT ErrorCode PrxSessionCreateSession(const char* name, int serverID, int worldID, int numSlots, const char* password, int creationFlags, int typeFlags, const char* sessionStatus);
	PRX_EXPORT ErrorCode PrxSessionCreateFriendsSession(const char* name, int serverID, int worldID, int numSlots, int numFriendSlots, const char* password, int creationFlags, const char* sessionStatus);

	// Session joining...
	PRX_EXPORT ErrorCode PrxSessionJoinSession(int sessionID, const char* password);
	PRX_EXPORT ErrorCode PrxSessionJoinInvitedSession();

	// Session Modification...
	PRX_EXPORT ErrorCode PrxSessionModifyClearAttributes();
	PRX_EXPORT ErrorCode PrxSessionModifyAddAttribute(ModifySessionAttribute* attribute);
	PRX_EXPORT ErrorCode PrxSessionModifySession(int attributeType);

	// Active session info...
	PRX_EXPORT ErrorCode PrxGetSessionInfo(SessionInfo* info);

	// Active session attributes...
	PRX_EXPORT void PrxLockSessionAttributeList();
	PRX_EXPORT void PrxUnlockSessionAttributeList();
	PRX_EXPORT int PrxGetSessionAttributeListCount();
	PRX_EXPORT ErrorCode PrxGetSessionAttributeInfo(int index, SessionAttributeInfo* info);

	// Active session member info...
	PRX_EXPORT void PrxLockSessionMemberList();
	PRX_EXPORT void PrxUnlockSessionMemberList();
	PRX_EXPORT int PrxGetSessionMemberListCount();
	PRX_EXPORT ErrorCode PrxGetSessionMemberInfo(int index, SessionMemberInfo* info);

	// Active session member attributes...
	PRX_EXPORT void PrxLockSessionMemberAttributeList();
	PRX_EXPORT void PrxUnlockSessionMemberAttributeList();
	PRX_EXPORT int PrxGetSessionMemberAttributeListCount(int memberIndex);
	PRX_EXPORT ErrorCode PrxGetSessionMemberAttributeInfo(int memberIndex, int index, SessionAttributeInfo* info);

	// Leaving a session...
	PRX_EXPORT ErrorCode PrxSessionLeaveSession();

	// Session searching...
	PRX_EXPORT ErrorCode PrxSessionFind(int serverID, int worldID);
	PRX_EXPORT ErrorCode PrxSessionFindFriends(int serverID, int worldID);
	PRX_EXPORT ErrorCode PrxSessionFindRegional(int serverID, int worldID);

	// Found session info...
	PRX_EXPORT void PrxLockFoundSessionList();
	PRX_EXPORT void PrxUnlockFoundSessionList();
	PRX_EXPORT int PrxGetFoundSessionListCount();
	PRX_EXPORT ErrorCode PrxGetFoundSessionInfo(int sessionIndex, SessionInfo* info);

	// Found session attributes
	PRX_EXPORT void PrxLockFoundSessionAttributeList();
	PRX_EXPORT void PrxUnlockFoundSessionAttributeList();
	PRX_EXPORT int PrxGetFoundSessionAttributeListCount(int sessionIndex);
	PRX_EXPORT ErrorCode PrxGetFoundSessionAttributeInfo(int sessionIndex, int attributeIndex, SessionAttributeInfo* info);

	// Session invites
	PRX_EXPORT ErrorCode PrxInviteToSession(const char* text, int ps4NpIdsCount);

	class CachedMatching
	{
		SimpleLock m_Lock;
		bool m_SessionBusy;
		ResultCode m_LastResult;

		std::vector<sce::Toolkit::NP::RegisterSessionAttribute> m_AttributeDefinitions;
		std::vector<sce::Toolkit::NP::SessionRequestAttribute> m_SessionAttributes;
		std::vector<sce::Toolkit::NP::ModifySessionAttributes> m_ModifyAttributes;

		sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::SessionInformation> m_FutureSessionProcess;
		sce::Toolkit::NP::Utilities::Future<int> m_FutureSessionResult;
		sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::SessionList>* m_FutureSearchProcess;
		//sce::Toolkit::NP::Utilities::Future<sce::Toolkit::NP::SearchSessionsRequest> m_FutureSearchCriteria;

		sce::Toolkit::NP::SessionInformation m_SessionInformation;
		sce::Toolkit::NP::SessionList m_SessionList;

	public:
		CachedMatching();
		bool ProcessEvent(const sce::Toolkit::NP::Event& event);

		ErrorCode ClearAttributeDefinitions();
		ErrorCode AddAttributeDefinition(const char* name, int type, int valueType, int maxSize);
		ErrorCode RegisterAttributeDefinitions();

		bool InSession();
		bool SessionIsBusy();
		ErrorCode GetLastError() const { return m_LastResult.GetResult(); }
		bool GetLastError(ResultCode* result) const { *result = m_LastResult; return m_LastResult.GetResult() == NP_OK; }

		ErrorCode ClearSessionAttributes();
		ErrorCode AddSessionAttribute(const SessionAttribute* attribute);
		ErrorCode CreateSession(const char* name, int serverID, int worldID, int numSlots, int numFriendSlots, const char* password, int creationFlags, int typeFlags, const char* sessionStatus);

		ErrorCode JoinSession(unsigned int sessionID, const char* password);
		ErrorCode JoinInvitedSession();
		ErrorCode JoinInvitedSessionWithMemberAttributes();

		ErrorCode ClearModifySessionAttributes();
		ErrorCode AddModifySessionAttribute(const ModifySessionAttribute* attribute);
		ErrorCode ModifySession(int attributeType);

		ErrorCode Search(int serverID, int worldID);
		ErrorCode SearchFriends(int serverID, int worldID);
		ErrorCode SearchRegional(int serverID, int worldID);
		ErrorCode SearchSession(int serverID, int worldID, int searchFlags);

		// Found session info.
		int GetSessionFoundListCount();
		ErrorCode GetSessionFoundInfo(unsigned int sessionIndex, SessionInfo* info);
		int GetSessionFoundAttributeCount(unsigned int sessionIndex);
		ErrorCode GetSessionFoundAttribute(unsigned int sessionIndex, unsigned int attributeIndex, SessionAttributeInfo* info);

		// Active session info.
		ErrorCode GetSessionInfo(SessionInfo* info);
		int GetSessionAttributeCount();
		ErrorCode GetSessionAttribute(unsigned int attributeIndex, SessionAttributeInfo* info);

		// Active session member info.
		int GetSessionMemberCount();
		ErrorCode GetSessionMemberInfo(unsigned int memberIndex, SessionMemberInfo* info);
		int GetSessionMemberAttributeCount(unsigned int memberIndex);
		ErrorCode GetSessionMemberAttribute(unsigned int memberIndex, unsigned int index, SessionAttributeInfo* info);

		// Session invites
		ErrorCode InviteToSession(const char* text, int npIdsCount);

		void Lock();
		void Unlock();

		ErrorCode LeaveSession();

	private:
		ErrorCode CreateSessionSearchObject();
		void DestroySessionSearchObject();
	};

	extern CachedMatching gMatching;
}

#endif // _SESSION_H
